// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_IRADIO_CALLBACKS_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_IRADIO_CALLBACKS_H

#include "common.h"

struct RdsDataStruct {
    vector<int32_t> alternativeFrequencyHz; /**< This is the AF field of RDS. */
    int32_t programIdentification; /**< This is the PI field of RDS. */
    string programService; /**< This is the PS field of RDS. */
    int32_t programType; /**< This is the PTY field of RDS. */
    string programTypeName; /**< This is the PTYN field of RDS. */
    string radioText; /**< This is the RT field of RDS. */
    int32_t musicSpeechSwitch; /**< This is the MS field of RDS. 0 = speech, 1 = music  */
    bool trafficProgramFlag; /**< This is the TP field of RDS. */
    bool trafficAnnouncementFlag; /**< This is the TA field of RDS. */
};

struct HdRadioDataStruct {
};

struct StationMetaDataStruct {
    int32_t audioChannels; /**< 0 = unknown, 1 = mono, 2 = stereo. */
    int32_t signalQuality; /**< 0 - 100 */
    RdsDataStruct *rds; /**< null to omit data */
    HdRadioDataStruct *hdRadioData;  /**< null to omit data */
};

struct StationInfoStruct {
    int32_t type; /**< see enum RadioType */
    int32_t channel; /**< For AM/FM: station frequency in hz. */
    int32_t subChannel;  /**< For HD: 1-8 */
    StationMetaDataStruct *metaData;  /**< null to omit data */
};

struct StationPresetStruct {
    int32_t type; /**< see enum RadioType */
    int32_t channel; /**< for am/fm, in hertz */
    int32_t subChannel; /**< for HD radio, 1-8 = HD1-HD8 */
};

struct StationPresetListStruct {
    string name; /** name of list */
    vector<int32_t> restrictedStationTypes; /** see RadioType */
    vector<StationPresetStruct> presets;
};

struct TrafficIncidentStruct {
    int32_t eventCode;
    int32_t expectedIncidentDuration; /**< in seconds */
    double longitude;
    double latitude;
};

struct RadioPropertiesStruct {
    int32_t radioId; /**< uniq non-zero id for this radio */
    int32_t radioType; /**< see enum RadioType */
    vector<pair<int32_t,int32_t> > channelRange; /**< list of start/end pairs that define the channel range */
    vector<int32_t> channelSpacings; /**< list of valid channel spacings */
    int32_t channelSpacing; /**< current channel spacing */
    int32_t backgroundTuner; /**< if a background tuner exists */
    int32_t region; /**< See enum ItuRegion, -1 to omit */
    int32_t rds; /**< See enum RdsType, -1 to omit */
    int32_t afSwitch; /**< -1 to omit */
    int32_t ta; /**< -1 to omit */
    int32_t trafficService; /**< see enum TrafficServiceType, -1 to omit */
    /**
     * -1 to omit, 0 = false, 1 = true
     *  When set, indicates radio audio is looped back to the SoC for capture.
     */
    int32_t audioLoopback;
    /**
     * -1 to omit, 0 = false, 1 = true
     * When set, indicates radio only muting is supported.
     */
    int32_t muteCapability;
    /** -1 to omit, 0 = no access, 1 = read only, 2 = read and write */
    int32_t stationPresetsAccess;
};

struct RadioStateStruct {
    bool radioSourceEnabled;
    int32_t radioMuted; /**< -1 to omit, 0 = false, 1 = true */
    int32_t activeRadioId;
    StationInfoStruct stationInfo;
    vector<StationInfoStruct> programList;
    vector<StationPresetListStruct> stationPresetLists;
};

struct RadioConfigurationStruct {
    vector<RadioPropertiesStruct> radioProperties;
};


/**
 * This class represents an interface that every sink implementation must subclass if they
 * wish to implement the part of the GAL protocol that allows for the control of an on-board
 * AM/FM tuner. You should also take a look at the class RadioEndpoint for a pseudocode
 * example.
 */
class IRadioCallbacks {
public:
    virtual ~IRadioCallbacks() { }
    /**
     * Gets invoked when the current radio state is requested.
     * @param radioState pointer storing the current radio state.
     */
    virtual void onGetCurrentRadioState(RadioStateStruct *radioState) = 0;
    /**
     * Gets invoked when the client has requested to tune to a station.
     * The host must call sendTuneToStationResponse() in response to the request
     * and sendRadioStationInfoNotification() when it's completed tuning
     * to the new station or if an error occurred.
     * @param radioId ID of the radio requested to do the tuning.
     * @param channel channel value (for am/fm, this is in hz).
     * @param subChannel applicable only for HD radio, 1-8 = HD1 to HD8.
     * @return STATUS_SUCCESS if call succeeded.
     */
    virtual int onTuneToStationRequest(int32_t radioId, int32_t channel, int32_t subChannel) = 0;
    /**
     * Gets invoked when the client has requested to step to the next channel.
     * The host must call sendStepChannelResponse() in response to the request
     * and sendRadioStationInfoNotification() when it's completed tuning
     * to the next channel.  Radio should wrap around if channel overflows or underflows.
     * @param radioId ID of the radio requested to do the stepping.
     * @param up step direction (up/down)
     * @param skipSubChannel applicable only for HD radio, if true, radio should
     * step to the next channel, if false, radio should step to the next sub
     * channel if available (i.e. from 95.7 HD1 to 95.7 HD2).
     * @return STATUS_SUCCESS if call succeeded.
     */
    virtual int onStepChannelRequest(int32_t radioId, bool up, bool skipSubChannel) = 0;
    /**
     * Gets invoked when the client has requested to seek to the next available station.
     * The host must call sendSeekStationResponse() in response to the request
     * and sendRadioStationInfoNotification() when it's found the new
     * station.  Radio should wrap around if channel overflows or underflows.
     * @param radioId ID of the radio requested to do the seeking.
     * @param up seek direction (up/down)
     * @param skipSubChannel applicable only for HD radio, if true, radio should
     * seek to the next available channel, if false, radio should seek to the next available sub
     * channel if available (i.e. from 95.7 HD1 to 95.7 HD2).
     * @return STATUS_SUCCESS if call succeeded.
     */
    virtual int onSeekStationRequest(int32_t radioId, bool up, bool skipSubChannel) = 0;
    /**
     * Gets invoked when the client has requested to scan through stations.
     * The host must call sendScanStationsResponse() in response to the request
     * and sendRadioStationInfoNotification() when ever a new station is
     * scanned.
     * Radio should wrap around if channel overflows or underflows.
     * @param radioId ID of the radio requested to do the scanning.
     * @param up scan direction (up/down)
     * @param skipSubChannel applicable only for HD radio, if true, radio should
     * scan to the next available channel, if false, radio should scan to the next available sub
     * channel if available (i.e. from 95.7 HD1 to 95.7 HD2).
     * @return STATUS_SUCCESS if call succeeded.
     */
    virtual int onScanStationsRequest(int32_t radioId, bool start, bool up, bool skipSubChannel) = 0;
    /**
     * Gets invoked when the client has requested to cancel any pending operations.
     * The host must call sendCancelRadioOperationsResponse() when it's finished
     * canceling pending operations.
     * @param radioId ID of the radio requested to cancel operations.
     * @return STATUS_SUCCESS if call succeeded.
     */
    virtual int onCancelRadioOperationsRequest(int32_t radioId) = 0;
    /**
     * Gets invoked when the client has requested to configure the channel spacing.
     * The host must call sendConfigureChannelSpacingResponse() when it's finished
     * processing the request.
     * @param radioId ID of the radio requested to cancel operations.
     * @return STATUS_SUCCESS if call succeeded.
     */
    virtual int onConfigureChannelSpacingRequest(int32_t radioId, int32_t channelSpacing) = 0;
    /**
     * Gets invoked when the client has requested to mute radio (radio only).
     * The host must call sendMuteRadioResponse() when it's finished processing
     * the request or if an error occurred.
     * @param radioId ID of radio to mute.
     * @param mute mute/unmute flag.
     * @return STATUS_SUCCESS if call succeeded.
     */
    virtual int onMuteRadioRequest(int32_t radioId, bool mute) = 0;
    /**
     * Gets invoked when the client has requested to select the active radio.
     * The host must call sendActiveRadioNotification() to notify the client of
     * the active radio change.
     * @param radio_id id of radio to select
     * @return STATUS_SUCCESS if call succeeded.
     */
    virtual int onSelectActiveRadioRequest(int32_t radio_id) = 0;
    /**
     * Gets invoked when the client has requested a traffic update from the host.
     * The host must call sendGetTrafficUpdateResponse() to send a traffic update response
     * back to the client.
     * @param radioId ID of radio to get traffic update from.
     * @return STATUS_SUCCESS if call succeeded.
     */
    virtual int onGetTrafficUpdateRequest(int32_t radioId) = 0;
    /**
     * Gets invoked when the client has requested to get a list of all the available
     * programs from the host.
     * The host must call sendGetProgramListResponse() to send the list of
     * discovered programs back to the client.  sendGetProgramListResponse() may
     * be called multiple times for a single request (see function for more
     * detail).
     * @param radioId ID of radio to get program list from.
     * @return STATUS_SUCCESS if call succeeded.
     */
    virtual int onGetProgramListRequest(int32_t radioId) = 0;
    /**
     * Gets invoked when the client has requested to select radio as the source.
     * The host must call sendRadioSourceResponse() in response to this.
     */
    virtual int onRadioSourceRequest() = 0;
};

#endif
